<?php
// Iniciar sesión si no está iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Incluir archivo de conexión
require 'connect.php';

// Verificar si el usuario está logueado
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Obtener información del usuario
$user_id = $_SESSION['user_id'];
$user = [];
$error = null;

try {
    // Consulta para obtener datos del usuario
    $stmt = $conn->prepare("SELECT * FROM usuarios WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        throw new Exception("Usuario no encontrado");
    }

    // Inicializar variables con valores por defecto
    $default_values = [
        'saldo_deposito' => 0,
        'total_retirado' => 0,
        'saldo_inversion' => 0,
        'saldo_planes' => 0
    ];

    foreach ($default_values as $key => $value) {
        if (!isset($user[$key])) {
            $user[$key] = $value;
        }
    }

    // Obtener estadísticas avanzadas
    $stats = [
        'total_invertido' => 0,
        'total_actual_invertido' => 0, // Nuevo campo para monto actual en planes
        'total_reinvertido' => 0,
        'total_ganado' => 0,
        'planes_activos' => 0,
        'planes_completados' => 0,
        'depositos_pendientes' => 0,
        'retiros_pendientes' => 0,
        'ultimo_deposito' => null,
        'ultimo_retiro' => null,
        'mejor_inversion' => null
    ];

    // Consultas para estadísticas
    $queries = [
        'total_invertido' => "SELECT COALESCE(SUM(monto_inicial), 0) as total FROM planes WHERE usuario_id = ?",
        'total_actual_invertido' => "SELECT COALESCE(SUM(monto_actual), 0) as total FROM planes WHERE usuario_id = ? AND completado = FALSE", // Nueva consulta para monto actual
        'total_reinvertido' => "SELECT COALESCE(SUM(monto_inicial), 0) as total FROM planes WHERE usuario_id = ? AND es_reinversion = TRUE",
        'total_ganado' => "SELECT COALESCE(SUM(ganancia_acumulada), 0) as total FROM planes WHERE usuario_id = ?",
        'planes_activos' => "SELECT COUNT(*) as total FROM planes WHERE usuario_id = ? AND completado = FALSE",
        'planes_completados' => "SELECT COUNT(*) as total FROM planes WHERE usuario_id = ? AND completado = TRUE",
        'depositos_pendientes' => "SELECT COUNT(*) as total FROM depositos WHERE usuario_id = ? AND estado = 'pendiente'",
        'retiros_pendientes' => "SELECT COUNT(*) as total FROM retiros WHERE usuario_id = ? AND estado = 'pendiente'",
        'ultimo_deposito' => "SELECT monto, fecha FROM depositos WHERE usuario_id = ? ORDER BY fecha DESC LIMIT 1",
        'ultimo_retiro' => "SELECT monto, fecha FROM retiros WHERE usuario_id = ? ORDER BY fecha DESC LIMIT 1",
        'mejor_inversion' => "SELECT monto_inicial, monto_actual, ganancia_acumulada FROM planes WHERE usuario_id = ? ORDER BY ganancia_acumulada DESC LIMIT 1"
    ];

    foreach ($queries as $key => $sql) {
        $stmt = $conn->prepare($sql);
        $stmt->execute([$user_id]);
        
        if (in_array($key, ['ultimo_deposito', 'ultimo_retiro', 'mejor_inversion'])) {
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats[$key] = $result ? $result : null;
        } else {
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats[$key] = $result['total'] ?? 0;
        }
    }

    // Calcular ROI (Return on Investment)
    $roi = ($user['saldo_deposito'] > 0) ? 
           (($stats['total_ganado'] + $user['total_retirado']) / $user['saldo_deposito'] * 100) : 0;

    // Datos para gráficos
    $chart_data = [
        'depositos_mensuales' => [],
        'retiros_mensuales' => [],
        'distribucion_inversiones' => [],
        'rendimiento_planes' => []
    ];

    // Últimos 6 meses de depósitos
    $stmt = $conn->prepare("
        SELECT DATE_FORMAT(fecha, '%Y-%m') as mes, COALESCE(SUM(monto), 0) as total 
        FROM depositos 
        WHERE usuario_id = ? AND estado = 'completado' 
        AND fecha >= DATE_SUB(CURRENT_DATE, INTERVAL 6 MONTH)
        GROUP BY mes 
        ORDER BY mes
    ");
    $stmt->execute([$user_id]);
    $chart_data['depositos_mensuales'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Últimos 6 meses de retiros
    $stmt = $conn->prepare("
        SELECT DATE_FORMAT(fecha, '%Y-%m') as mes, COALESCE(SUM(monto), 0) as total 
        FROM retiros 
        WHERE usuario_id = ? AND estado = 'completado' 
        AND fecha >= DATE_SUB(CURRENT_DATE, INTERVAL 6 MONTH)
        GROUP BY mes 
        ORDER BY mes
    ");
    $stmt->execute([$user_id]);
    $chart_data['retiros_mensuales'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Distribución de inversiones (basado en monto_actual)
    $stmt = $conn->prepare("
        SELECT 
            CASE 
                WHEN monto_actual < 100 THEN 'Menos de $100'
                WHEN monto_actual BETWEEN 100 AND 500 THEN '$100 - $500'
                WHEN monto_actual BETWEEN 501 AND 1000 THEN '$501 - $1,000'
                WHEN monto_actual BETWEEN 1001 AND 5000 THEN '$1,001 - $5,000'
                ELSE 'Más de $5,000'
            END as rango,
            COUNT(*) as cantidad,
            COALESCE(SUM(monto_actual), 0) as total
        FROM planes
        WHERE usuario_id = ? AND completado = FALSE
        GROUP BY rango
        ORDER BY total DESC
    ");
    $stmt->execute([$user_id]);
    $chart_data['distribucion_inversiones'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Rendimiento de planes (basado en monto_actual)
    $stmt = $conn->prepare("
        SELECT 
            DATE_FORMAT(fecha_inicio, '%Y-%m') as mes,
            COUNT(*) as cantidad_planes,
            COALESCE(SUM(ganancia_acumulada), 0) as ganancia_total,
            COALESCE(SUM(monto_actual), 0) as inversion_total
        FROM planes
        WHERE usuario_id = ? AND completado = FALSE
        GROUP BY mes
        ORDER BY mes DESC
        LIMIT 6
    ");
    $stmt->execute([$user_id]);
    $chart_data['rendimiento_planes'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Actividad reciente
    $stmt = $conn->prepare("
        (SELECT 'deposito' as tipo, id, monto, estado, fecha, NULL as monto_inicial, NULL as monto_actual, NULL as porcentaje_completado, NULL as es_reinversion 
         FROM depositos WHERE usuario_id = ? ORDER BY fecha DESC LIMIT 5)
        UNION ALL
        (SELECT 'plan' as tipo, id, NULL as monto, NULL as estado, fecha_inicio as fecha, monto_inicial, monto_actual, porcentaje_completado, es_reinversion 
         FROM planes WHERE usuario_id = ? ORDER BY fecha_inicio DESC LIMIT 5)
        UNION ALL
        (SELECT 'retiro' as tipo, id, monto, estado, fecha, NULL as monto_inicial, NULL as monto_actual, NULL as porcentaje_completado, NULL as es_reinversion 
         FROM retiros WHERE usuario_id = ? ORDER BY fecha DESC LIMIT 5)
        ORDER BY fecha DESC LIMIT 10
    ");
    $stmt->execute([$user_id, $user_id, $user_id]);
    $actividades_recientes = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    $error = "Error en la base de datos: " . $e->getMessage();
    error_log($error);
} catch (Exception $e) {
    $error = $e->getMessage();
    error_log($error);
}

// Función para formatear fechas
function formatFecha($fecha) {
    if (empty($fecha)) return '';
    try {
        $date = new DateTime($fecha);
        return $date->format('d/m/Y H:i');
    } catch (Exception $e) {
        return $fecha;
    }
}

// Función para calcular el tiempo transcurrido
function tiempoTranscurrido($fecha) {
    if (empty($fecha)) return 'Reciente';
    
    try {
        $ahora = new DateTime();
        $fechaActividad = new DateTime($fecha);
        $diferencia = $ahora->diff($fechaActividad);
        
        if ($diferencia->y > 0) {
            return 'Hace ' . $diferencia->y . ' año' . ($diferencia->y > 1 ? 's' : '');
        } elseif ($diferencia->m > 0) {
            return 'Hace ' . $diferencia->m . ' mes' . ($diferencia->m > 1 ? 'es' : '');
        } elseif ($diferencia->d > 0) {
            return 'Hace ' . $diferencia->d . ' día' . ($diferencia->d > 1 ? 's' : '');
        } elseif ($diferencia->h > 0) {
            return 'Hace ' . $diferencia->h . ' hora' . ($diferencia->h > 1 ? 's' : '');
        } elseif ($diferencia->i > 0) {
            return 'Hace ' . $diferencia->i . ' minuto' . ($diferencia->i > 1 ? 's' : '');
        } else {
            return 'Hace unos segundos';
        }
    } catch (Exception $e) {
        return 'Reciente';
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>JC Investment - Dashboard</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-datalabels@2.0.0"></script>
    <style>
        /* Estilos CSS permanecen iguales */
        :root {
            --primary: #00ff9d;
            --secondary: #00b8ff;
            --dark: #121212;
            --darker: #0a0a0a;
            --light: #f0f0f0;
            --danger: #ff3860;
            --success: #00d1b2;
            --warning: #ffdd57;
            --info: #209cee;
            --purple: #9c27b0;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        body {
            background-color: var(--dark);
            color: var(--light);
            display: flex;
            min-height: 100vh;
        }
        
        .sidebar {
            width: 250px;
            background-color: rgba(10, 10, 10, 0.8);
            border-right: 1px solid rgba(0, 255, 157, 0.1);
            padding: 1.5rem;
            position: fixed;
            height: 100vh;
            z-index: 100;
        }
        
        .logo {
            text-align: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid rgba(0, 255, 157, 0.1);
        }
        
        .logo h2 {
            font-size: 1.5rem;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            -webkit-background-clip: text;
            background-clip: text;
            color: transparent;
            text-transform: uppercase;
            letter-spacing: 2px;
        }
        
        .nav-menu {
            list-style: none;
        }
        
        .nav-item {
            margin-bottom: 0.5rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 0.8rem 1rem;
            color: var(--light);
            text-decoration: none;
            border-radius: 5px;
            transition: all 0.3s ease;
        }
        
        .nav-link:hover, .nav-link.active {
            background-color: rgba(0, 255, 157, 0.1);
            color: var(--primary);
        }
        
        .nav-link i {
            margin-right: 0.8rem;
            font-size: 1.2rem;
        }
        
        .main-content {
            flex: 1;
            margin-left: 250px;
            padding: 2rem;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid rgba(0, 255, 157, 0.1);
        }
        
        .user-info {
            display: flex;
            align-items: center;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: rgba(0, 255, 157, 0.1);
            margin-right: 1rem;
            overflow: hidden;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .user-name {
            font-size: 1.2rem;
            font-weight: bold;
        }
        
        .logout-btn {
            background-color: rgba(255, 56, 96, 0.1);
            color: var(--danger);
            border: none;
            padding: 0.5rem 1rem;
            border-radius: 5px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .logout-btn:hover {
            background-color: rgba(255, 56, 96, 0.2);
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background-color: rgba(10, 10, 10, 0.8);
            border: 1px solid rgba(0, 255, 157, 0.1);
            border-radius: 10px;
            padding: 1.5rem;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0, 255, 157, 0.1);
            border-color: rgba(0, 255, 157, 0.3);
        }
        
        .stat-title {
            font-size: 0.9rem;
            color: rgba(255, 255, 255, 0.7);
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
        }
        
        .stat-title i {
            margin-right: 0.5rem;
        }
        
        .stat-value {
            font-size: 1.8rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            -webkit-background-clip: text;
            background-clip: text;
            color: transparent;
        }
        
        .stat-desc {
            font-size: 0.8rem;
            color: rgba(255, 255, 255, 0.5);
        }
        
        .stat-badge {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background-color: rgba(255, 255, 255, 0.1);
            padding: 0.2rem 0.5rem;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: bold;
        }
        
        .card-primary {
            border-top: 3px solid var(--primary);
        }
        
        .card-secondary {
            border-top: 3px solid var(--secondary);
        }
        
        .card-success {
            border-top: 3px solid var(--success);
        }
        
        .card-warning {
            border-top: 3px solid var(--warning);
        }
        
        .card-info {
            border-top: 3px solid var(--info);
        }
        
        .card-danger {
            border-top: 3px solid var(--danger);
        }
        
        .card-purple {
            border-top: 3px solid var(--purple);
        }
        
        .charts-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .chart-card {
            background-color: rgba(10, 10, 10, 0.8);
            border: 1px solid rgba(0, 255, 157, 0.1);
            border-radius: 10px;
            padding: 1.5rem;
            transition: all 0.3s ease;
        }
        
        .chart-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0, 255, 157, 0.1);
            border-color: rgba(0, 255, 157, 0.3);
        }
        
        .chart-container {
            position: relative;
            height: 300px;
            width: 100%;
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .info-card {
            background-color: rgba(10, 10, 10, 0.8);
            border: 1px solid rgba(0, 255, 157, 0.1);
            border-radius: 10px;
            padding: 1.5rem;
        }
        
        .info-card-header {
            display: flex;
            align-items: center;
            margin-bottom: 1rem;
            padding-bottom: 0.5rem;
            border-bottom: 1px solid rgba(0, 255, 157, 0.1);
        }
        
        .info-card-header i {
            margin-right: 0.8rem;
            font-size: 1.2rem;
            color: var(--primary);
        }
        
        .info-card-title {
            font-size: 1rem;
            font-weight: bold;
        }
        
        .info-item {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
            padding: 0.5rem 0;
        }
        
        .info-item:not(:last-child) {
            border-bottom: 1px dashed rgba(255, 255, 255, 0.05);
        }
        
        .info-label {
            color: rgba(255, 255, 255, 0.7);
            display: flex;
            align-items: center;
        }
        
        .info-label i {
            margin-right: 0.5rem;
            font-size: 0.9rem;
        }
        
        .info-value {
            font-weight: bold;
            color: var(--primary);
        }
        
        .progress-container {
            margin-top: 1rem;
        }
        
        .progress-title {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
            font-size: 0.8rem;
            color: rgba(255, 255, 255, 0.7);
        }
        
        .progress-bar {
            height: 6px;
            background-color: rgba(255, 255, 255, 0.1);
            border-radius: 3px;
            overflow: hidden;
        }
        
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            border-radius: 3px;
            transition: width 0.5s ease;
        }
        
        .recent-activity {
            background-color: rgba(10, 10, 10, 0.8);
            border: 1px solid rgba(0, 255, 157, 0.1);
            border-radius: 10px;
            padding: 1.5rem;
            margin-top: 2rem;
        }
        
        .section-title {
            font-size: 1.2rem;
            margin-bottom: 1rem;
            color: var(--primary);
            display: flex;
            align-items: center;
        }
        
        .section-title i {
            margin-right: 0.5rem;
        }
        
        .activity-list {
            list-style: none;
        }
        
        .activity-item {
            padding: 1rem 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.05);
            display: flex;
            align-items: center;
            transition: all 0.3s ease;
        }
        
        .activity-item:hover {
            background-color: rgba(0, 255, 157, 0.05);
        }
        
        .activity-item:last-child {
            border-bottom: none;
        }
        
        .activity-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: rgba(0, 255, 157, 0.1);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 1rem;
            color: var(--primary);
            flex-shrink: 0;
        }
        
        .activity-content {
            flex: 1;
            min-width: 0;
        }
        
        .activity-title {
            font-size: 0.9rem;
            margin-bottom: 0.2rem;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .activity-desc {
            font-size: 0.8rem;
            color: rgba(255, 255, 255, 0.6);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .activity-time {
            font-size: 0.7rem;
            color: rgba(255, 255, 255, 0.4);
            margin-left: 1rem;
            white-space: nowrap;
        }
        
        .glow-effect {
            position: relative;
            overflow: hidden;
        }
        
        .glow-effect::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: linear-gradient(
                to bottom right,
                rgba(0, 255, 157, 0),
                rgba(0, 255, 157, 0),
                rgba(0, 255, 157, 0.3)
            );
            transform: rotate(30deg);
            animation: glow 6s linear infinite;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .glow-effect:hover::before {
            opacity: 0.3;
        }
        
        @keyframes glow {
            0% {
                transform: rotate(30deg) translate(-10%, -10%);
            }
            100% {
                transform: rotate(30deg) translate(10%, 10%);
            }
        }
        
        .no-activity {
            text-align: center;
            padding: 1rem;
            color: rgba(255, 255, 255, 0.5);
            font-size: 0.9rem;
        }
        
        .error-message {
            background-color: rgba(255, 56, 96, 0.1);
            color: var(--danger);
            padding: 1rem;
            border-radius: 5px;
            margin-bottom: 1rem;
            border-left: 4px solid var(--danger);
            display: flex;
            align-items: center;
        }
        
        .error-message i {
            margin-right: 0.5rem;
        }
        
        .quick-actions {
            display: flex;
            gap: 1rem;
            margin-bottom: 2rem;
            flex-wrap: wrap;
        }
        
        .quick-action-btn {
            background-color: rgba(0, 255, 157, 0.1);
            color: var(--primary);
            border: none;
            padding: 0.8rem 1.2rem;
            border-radius: 5px;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            font-size: 0.9rem;
            flex-grow: 1;
            justify-content: center;
        }
        
        .quick-action-btn:hover {
            background-color: rgba(0, 255, 157, 0.2);
            transform: translateY(-2px);
        }
        
        .quick-action-btn i {
            margin-right: 0.5rem;
        }
        
        .badge {
            display: inline-block;
            padding: 0.2rem 0.5rem;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: bold;
            margin-left: 0.5rem;
        }
        
        .badge-success {
            background-color: rgba(0, 209, 178, 0.1);
            color: var(--success);
        }
        
        .badge-warning {
            background-color: rgba(255, 221, 87, 0.1);
            color: var(--warning);
        }
        
        .badge-danger {
            background-color: rgba(255, 56, 96, 0.1);
            color: var(--danger);
        }
        
        .badge-info {
            background-color: rgba(32, 156, 238, 0.1);
            color: var(--info);
        }
        
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
                padding: 1rem;
            }
            
            .main-content {
                margin-left: 0;
                padding: 1rem;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .charts-container {
                grid-template-columns: 1fr;
            }
            
            .info-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="sidebar">
        <div class="logo">
            <h2>JC Investment</h2>
        </div>
        
        <ul class="nav-menu">
            <li class="nav-item">
                <a href="dashboard.php" class="nav-link active">
                    <i class="fas fa-home"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="perfil.php" class="nav-link">
                    <i class="fas fa-user"></i>
                    <span>Perfil</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="planes.php" class="nav-link">
                    <i class="fas fa-chart-line"></i>
                    <span>Planes</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="transferencia.php" class="nav-link">
                    <i class="fas fa-exchange-alt"></i>
                    <span>Transferencia</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="historial.php" class="nav-link">
                    <i class="fas fa-history"></i>
                    <span>Historial</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="deposito.php" class="nav-link">
                    <i class="fas fa-wallet"></i>
                    <span>Depósito</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="retiro.php" class="nav-link">
                    <i class="fas fa-money-bill-wave"></i>
                    <span>Retiro</span>
                </a>
            </li>
        </ul>
    </div>
    
    <div class="main-content">
        <div class="header">
            <div class="user-info">
                <div class="user-avatar">
                    <img src="<?php echo isset($user['foto_perfil']) ? htmlspecialchars($user['foto_perfil']) : 'default.png'; ?>" alt="Foto de perfil">
                </div>
                <div class="user-name"><?php echo isset($user['nombre_completo']) ? htmlspecialchars($user['nombre_completo']) : 'Usuario'; ?></div>
            </div>
            <button class="logout-btn" onclick="window.location.href='logout.php'">
                <i class="fas fa-sign-out-alt"></i> Cerrar Sesión
            </button>
        </div>

        <?php if ($error): ?>
            <div class="error-message">
                <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>
        
        <div class="quick-actions">
            <button class="quick-action-btn" onclick="window.location.href='deposito.php'">
                <i class="fas fa-plus-circle"></i> Nuevo Depósito
            </button>
            <button class="quick-action-btn" onclick="window.location.href='retiro.php'">
                <i class="fas fa-money-bill-wave"></i> Solicitar Retiro
            </button>
            <button class="quick-action-btn" onclick="window.location.href='planes.php'">
                <i class="fas fa-chart-line"></i> Ver Planes
            </button>
            <button class="quick-action-btn" onclick="window.location.href='transferencia.php'">
                <i class="fas fa-exchange-alt"></i> Transferir Fondos
            </button>
        </div>
        
        <div class="stats-grid">
            <div class="stat-card card-primary glow-effect">
                <div class="stat-title">
                    <i class="fas fa-wallet"></i> Total Depositado
                </div>
                <div class="stat-value">$<?php echo number_format($user['saldo_deposito'], 2); ?></div>
                <div class="stat-desc">Fondos totales depositados</div>
                <?php if ($stats['ultimo_deposito']): ?>
                    <div class="stat-desc">Último: $<?php echo number_format($stats['ultimo_deposito']['monto'], 2); ?></div>
                <?php endif; ?>
            </div>
            
            <div class="stat-card card-success glow-effect">
                <div class="stat-title">
                    <i class="fas fa-chart-line"></i> Total Invertido
                </div>
                <div class="stat-value">$<?php echo number_format($stats['total_actual_invertido'], 2); ?></div>
                
                <div class="stat-desc"><?php echo $stats['planes_activos']; ?> activos</div>
            </div>
            
            <div class="stat-card card-info glow-effect">
                <div class="stat-title">
                    <i class="fas fa-coins"></i> Total Ganado
                </div>
                <div class="stat-value">$<?php echo number_format($stats['total_ganado'], 2); ?></div>
                <div class="stat-desc">Ganancias acumuladas</div>
                <?php if ($stats['mejor_inversion']): ?>
                    <div class="stat-desc">Mejor: $<?php echo number_format($stats['mejor_inversion']['ganancia_acumulada'], 2); ?></div>
                <?php endif; ?>
            </div>
            
            <div class="stat-card card-warning glow-effect">
                <div class="stat-title">
                    <i class="fas fa-money-bill-wave"></i> Total Retirado
                </div>
                <div class="stat-value">$<?php echo number_format($user['total_retirado'], 2); ?></div>
                <div class="stat-desc">Fondos retirados</div>
                <?php if ($stats['ultimo_retiro']): ?>
                    <div class="stat-desc">Último: $<?php echo number_format($stats['ultimo_retiro']['monto'], 2); ?></div>
                <?php endif; ?>
            </div>
            
           
            
            <div class="stat-card card-purple glow-effect">
                <div class="stat-title">
                    <i class="fas fa-percentage"></i> ROI
                </div>
                <div class="stat-value"><?php echo round($roi, 2); ?>%</div>
                <div class="stat-desc">Retorno sobre inversión</div>
                <div class="stat-desc">$1 → $<?php echo number_format(1 + ($roi / 100), 2); ?></div>
            </div>
            
            <div class="stat-card card-danger glow-effect">
                <div class="stat-title">
                    <i class="fas fa-clock"></i> Pendientes
                </div>
                <div class="stat-value"><?php echo $stats['depositos_pendientes'] + $stats['retiros_pendientes']; ?></div>
                <div class="stat-desc">
                    <span class="badge badge-warning"><?php echo $stats['depositos_pendientes']; ?> depósitos</span>
                    <span class="badge badge-danger"><?php echo $stats['retiros_pendientes']; ?> retiros</span>
                </div>
            </div>
            
            <div class="stat-card card-success glow-effect">
                <div class="stat-title">
                    <i class="fas fa-piggy-bank"></i> Disponible Para Retirar
                </div>
                <div class="stat-value">$<?php echo number_format($user['saldo_planes'], 2); ?></div>
                <div class="stat-desc">Para retiro inmediato</div>
                <div class="stat-desc"><?php echo round($user['saldo_planes'] / max($stats['total_ganado'], 1) * 100); ?>% de ganancias</div>
            </div>
        </div>
        
        <div class="info-grid">
            <div class="info-card">
                <div class="info-card-header">
                    <i class="fas fa-chart-pie"></i>
                    <div class="info-card-title">Resumen de Inversiones</div>
                </div>
                
                <div class="info-item">
                    <span class="info-label"><i class="fas fa-wallet"></i> Saldo Disponible:</span>
                    <span class="info-value">$<?php echo number_format($user['saldo_inversion'], 2); ?></span>
                </div>
                
                <div class="info-item">
                    <span class="info-label"><i class="fas fa-play-circle"></i> Planes Activos:</span>
                    <span class="info-value"><?php echo $stats['planes_activos']; ?></span>
                </div>
                
                <div class="info-item">
                    <span class="info-label"><i class="fas fa-check-circle"></i> Planes Completados:</span>
                    <span class="info-value"><?php echo $stats['planes_completados']; ?></span>
                </div>
                
                <div class="info-item">
                    <span class="info-label"><i class="fas fa-money-bill-wave"></i> Monto Actual en Planes:</span>
                    <span class="info-value">$<?php echo number_format($stats['total_actual_invertido'], 2); ?></span>
                </div>
                
                <div class="progress-container">
                    <div class="progress-title">
                        <span>Progreso de inversión</span>
                        <span><?php 
                            $total = $stats['total_invertido'] + $user['saldo_inversion'];
                            echo $total > 0 ? round(($stats['total_invertido'] / $total * 100)) : 0;
                        ?>%</span>
                    </div>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?php 
                            echo $total > 0 ? ($stats['total_invertido'] / $total * 100) : 0;
                        ?>%"></div>
                    </div>
                </div>
            </div>
            
            <div class="info-card">
                <div class="info-card-header">
                    <i class="fas fa-wallet"></i>
                    <div class="info-card-title">Balance General</div>
                </div>
                
                <div class="info-item">
                    <span class="info-label"><i class="fas fa-arrow-down"></i> Depósitos Totales:</span>
                    <span class="info-value">$<?php echo number_format($user['saldo_deposito'], 2); ?></span>
                </div>
                
                <div class="info-item">
                    <span class="info-label"><i class="fas fa-arrow-up"></i> Retiros Totales:</span>
                    <span class="info-value">$<?php echo number_format($user['total_retirado'], 2); ?></span>
                </div>
                
                <div class="info-item">
                    <span class="info-label"><i class="fas fa-coins"></i> Ganancias Acumuladas:</span>
                    <span class="info-value">$<?php echo number_format($stats['total_ganado'], 2); ?></span>
                </div>
                
                <div class="info-item">
                    <span class="info-label"><i class="fas fa-hand-holding-usd"></i> Disponible para Retirar:</span>
                    <span class="info-value">$<?php echo number_format($user['saldo_planes'], 2); ?></span>
                </div>
                
                <div class="progress-container">
                    <div class="progress-title">
                        <span>Rendimiento total</span>
                        <span><?php 
                            echo $user['saldo_deposito'] > 0 ? 
                                round((($stats['total_ganado'] + $user['total_retirado']) / $user['saldo_deposito'] * 100)) : 0;
                        ?>%</span>
                    </div>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?php 
                            echo $user['saldo_deposito'] > 0 ? 
                                (($stats['total_ganado'] + $user['total_retirado']) / $user['saldo_deposito'] * 100) : 0;
                        ?>%"></div>
                    </div>
                </div>
            </div>
        </div>
        
        
        
        <div class="recent-activity">
            <h3 class="section-title">
                <i class="fas fa-bell"></i> Actividad Reciente
            </h3>
            
            <ul class="activity-list">
                <?php if (empty($actividades_recientes)): ?>
                    <li class="no-activity">No hay actividad reciente para mostrar</li>
                <?php else: ?>
                    <?php foreach ($actividades_recientes as $actividad): ?>
                        <li class="activity-item">
                            <div class="activity-icon" style="color: <?php 
                                if ($actividad['tipo'] == 'deposito') {
                                    echo $actividad['estado'] == 'completado' ? 'var(--success)' : 
                                         ($actividad['estado'] == 'pendiente' ? 'var(--warning)' : 'var(--danger)');
                                } elseif ($actividad['tipo'] == 'plan') {
                                    echo $actividad['es_reinversion'] ? 'var(--info)' : 'var(--primary)';
                                } else {
                                    echo $actividad['estado'] == 'completado' ? 'var(--success)' : 
                                         ($actividad['estado'] == 'pendiente' ? 'var(--warning)' : 'var(--danger)');
                                }
                            ?>">
                                <i class="<?php 
                                    if ($actividad['tipo'] == 'deposito') echo 'fas fa-wallet';
                                    elseif ($actividad['tipo'] == 'plan') echo $actividad['es_reinversion'] ? 'fas fa-redo' : 'fas fa-chart-line';
                                    else echo 'fas fa-money-bill-wave';
                                ?>"></i>
                            </div>
                            <div class="activity-content">
                                <h4 class="activity-title">
                                    <?php 
                                        if ($actividad['tipo'] == 'deposito') {
                                            echo 'Depósito ' . ucfirst($actividad['estado']);
                                        } elseif ($actividad['tipo'] == 'plan') {
                                            echo ($actividad['es_reinversion'] ? 'Reinversión' : 'Inversión') . ' iniciada';
                                        } else {
                                            echo 'Retiro ' . ucfirst($actividad['estado']);
                                        }
                                    ?>
                                </h4>
                                <p class="activity-desc">
                                    <?php 
                                        if ($actividad['tipo'] == 'deposito') {
                                            echo 'Monto: $' . number_format($actividad['monto'], 2) . ' USDT';
                                        } elseif ($actividad['tipo'] == 'plan') {
                                            echo 'Monto: $' . number_format($actividad['monto_inicial'], 2) . ' USDT';
                                            if ($actividad['porcentaje_completado']) {
                                                echo ' - ' . $actividad['porcentaje_completado'] . '% completado';
                                            }
                                            echo ' - Actual: $' . number_format($actividad['monto_actual'], 2);
                                        } else {
                                            echo 'Monto: $' . number_format($actividad['monto'], 2) . ' USDT';
                                        }
                                    ?>
                                </p>
                            </div>
                            <div class="activity-time"><?php echo tiempoTranscurrido($actividad['fecha']); ?></div>
                        </li>
                    <?php endforeach; ?>
                <?php endif; ?>
            </ul>
        </div>
    </div>

    <script>
        // Configuración común para los gráficos
        Chart.defaults.color = '#f0f0f0';
        Chart.defaults.borderColor = 'rgba(255, 255, 255, 0.1)';
        
        // Gráfico de depósitos vs retiros
        document.addEventListener('DOMContentLoaded', function() {
            const movimientosCtx = document.getElementById('movimientosChart');
            if (movimientosCtx) {
                const meses = [];
                const ahora = new Date();
                
                // Generar etiquetas para los últimos 6 meses
                for (let i = 5; i >= 0; i--) {
                    const fecha = new Date(ahora.getFullYear(), ahora.getMonth() - i, 1);
                    meses.push(fecha.toLocaleDateString('es-ES', { month: 'short', year: 'numeric' }));
                }
                
                // Preparar datos para depósitos y retiros
                const datosDepositos = Array(6).fill(0);
                const datosRetiros = Array(6).fill(0);
                
                <?php 
                // Mapear datos de PHP a JavaScript
                foreach ($chart_data['depositos_mensuales'] as $dep) {
                    echo "const depMes = new Date('{$dep['mes']}-01').toLocaleDateString('es-ES', { month: 'short', year: 'numeric' });\n";
                    echo "const indexDep = meses.indexOf(depMes);\n";
                    echo "if (indexDep !== -1) datosDepositos[indexDep] = {$dep['total']};\n";
                }
                
                foreach ($chart_data['retiros_mensuales'] as $ret) {
                    echo "const retMes = new Date('{$ret['mes']}-01').toLocaleDateString('es-ES', { month: 'short', year: 'numeric' });\n";
                    echo "const indexRet = meses.indexOf(retMes);\n";
                    echo "if (indexRet !== -1) datosRetiros[indexRet] = {$ret['total']};\n";
                }
                ?>
                
                new Chart(movimientosCtx, {
                    type: 'line',
                    data: {
                        labels: meses,
                        datasets: [
                            {
                                label: 'Depósitos',
                                data: datosDepositos,
                                borderColor: 'rgba(0, 209, 178, 1)',
                                backgroundColor: 'rgba(0, 209, 178, 0.1)',
                                tension: 0.3,
                                fill: true
                            },
                            {
                                label: 'Retiros',
                                data: datosRetiros,
                                borderColor: 'rgba(255, 221, 87, 1)',
                                backgroundColor: 'rgba(255, 221, 87, 0.1)',
                                tension: 0.3,
                                fill: true
                            }
                        ]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'top',
                                labels: {
                                    color: '#f0f0f0'
                                }
                            },
                            tooltip: {
                                mode: 'index',
                                intersect: false,
                                callbacks: {
                                    label: function(context) {
                                        return context.dataset.label + ': $' + context.raw.toFixed(2) + ' USDT';
                                    }
                                }
                            }
                        },
                        scales: {
                            x: {
                                grid: {
                                    color: 'rgba(255, 255, 255, 0.1)'
                                },
                                ticks: {
                                    color: '#f0f0f0'
                                }
                            },
                            y: {
                                grid: {
                                    color: 'rgba(255, 255, 255, 0.1)'
                                },
                                ticks: {
                                    color: '#f0f0f0',
                                    callback: function(value) {
                                        return '$' + value.toFixed(2);
                                    }
                                }
                            }
                        },
                        interaction: {
                            mode: 'nearest',
                            axis: 'x',
                            intersect: false
                        }
                    }
                });
            }
            
            // Gráfico de distribución de inversiones
            const distribucionCtx = document.getElementById('distribucionChart');
            if (distribucionCtx) {
                const labels = [];
                const data = [];
                const backgroundColors = [
                    'rgba(0, 255, 157, 0.8)',
                    'rgba(0, 184, 255, 0.8)',
                    'rgba(0, 209, 178, 0.8)',
                    'rgba(255, 221, 87, 0.8)',
                    'rgba(255, 56, 96, 0.8)'
                ];
                
                <?php 
                if (!empty($chart_data['distribucion_inversiones'])) {
                    foreach ($chart_data['distribucion_inversiones'] as $dist) {
                        echo "labels.push('" . addslashes($dist['rango']) . "');\n";
                        echo "data.push(" . $dist['total'] . ");\n";
                    }
                }
                ?>
                
                new Chart(distribucionCtx, {
                    type: 'doughnut',
                    data: {
                        labels: labels,
                        datasets: [{
                            data: data,
                            backgroundColor: backgroundColors.slice(0, labels.length),
                            borderColor: 'rgba(10, 10, 10, 0.8)',
                            borderWidth: 2
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'right',
                                labels: {
                                    color: '#f0f0f0'
                                }
                            },
                            tooltip: {
                                callbacks: {
                                    label: function(context) {
                                        const label = context.label || '';
                                        const value = context.raw || 0;
                                        const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                        const percentage = Math.round((value / total) * 100);
                                        return `${label}: $${value.toFixed(2)} (${percentage}%)`;
                                    }
                                }
                            },
                            datalabels: {
                                display: false
                            }
                        },
                        cutout: '70%'
                    },
                    plugins: [ChartDataLabels]
                });
            }
            
            // Gráfico de rendimiento de planes
            const rendimientoCtx = document.getElementById('rendimientoChart');
            if (rendimientoCtx) {
                const meses = [];
                const ganancias = [];
                const inversiones = [];
                const ahora = new Date();
                
                // Generar etiquetas para los últimos 6 meses
                for (let i = 5; i >= 0; i--) {
                    const fecha = new Date(ahora.getFullYear(), ahora.getMonth() - i, 1);
                    meses.push(fecha.toLocaleDateString('es-ES', { month: 'short', year: 'numeric' }));
                    ganancias.push(0);
                    inversiones.push(0);
                }
                
                <?php 
                if (!empty($chart_data['rendimiento_planes'])) {
                    foreach ($chart_data['rendimiento_planes'] as $plan) {
                        echo "const planMes = new Date('{$plan['mes']}-01').toLocaleDateString('es-ES', { month: 'short', year: 'numeric' });\n";
                        echo "const indexPlan = meses.indexOf(planMes);\n";
                        echo "if (indexPlan !== -1) {\n";
                        echo "  ganancias[indexPlan] = {$plan['ganancia_total']};\n";
                        echo "  inversiones[indexPlan] = {$plan['inversion_total']};\n";
                        echo "}\n";
                    }
                }
                ?>
                
                new Chart(rendimientoCtx, {
                    type: 'bar',
                    data: {
                        labels: meses,
                        datasets: [
                            {
                                label: 'Ganancias',
                                data: ganancias,
                                backgroundColor: 'rgba(0, 255, 157, 0.6)',
                                borderColor: 'rgba(0, 255, 157, 1)',
                                borderWidth: 1
                            },
                            {
                                label: 'Inversiones',
                                data: inversiones,
                                backgroundColor: 'rgba(0, 184, 255, 0.6)',
                                borderColor: 'rgba(0, 184, 255, 1)',
                                borderWidth: 1
                            }
                        ]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'top',
                                labels: {
                                    color: '#f0f0f0'
                                }
                            },
                            tooltip: {
                                callbacks: {
                                    label: function(context) {
                                        return context.dataset.label + ': $' + context.raw.toFixed(2) + ' USDT';
                                    }
                                }
                            }
                        },
                        scales: {
                            x: {
                                stacked: false,
                                grid: {
                                    color: 'rgba(255, 255, 255, 0.1)'
                                }
                            },
                            y: {
                                stacked: false,
                                grid: {
                                    color: 'rgba(255, 255, 255, 0.1)'
                                },
                                ticks: {
                                    callback: function(value) {
                                        return '$' + value.toFixed(2);
                                    }
                                }
                            }
                        }
                    }
                });
            }
            
            // Gráfico de evolución de saldos
            const saldosCtx = document.getElementById('saldosChart');
            if (saldosCtx) {
                const meses = [];
                const ahora = new Date();
                
                // Generar etiquetas para los últimos 6 meses
                for (let i = 5; i >= 0; i--) {
                    const fecha = new Date(ahora.getFullYear(), ahora.getMonth() - i, 1);
                    meses.push(fecha.toLocaleDateString('es-ES', { month: 'short', year: 'numeric' }));
                }
                
                // Datos de ejemplo (en un sistema real, estos vendrían de la base de datos)
                const saldosDeposito = [500, 1200, 1800, 2200, 3000, 3500];
                const saldosInversion = [300, 800, 1200, 1500, 2000, 2500];
                const saldosGanancias = [50, 150, 300, 500, 800, 1200];
                
                new Chart(saldosCtx, {
                    type: 'line',
                    data: {
                        labels: meses,
                        datasets: [
                            {
                                label: 'Saldo Depósito',
                                data: saldosDeposito,
                                borderColor: 'rgba(0, 209, 178, 1)',
                                backgroundColor: 'rgba(0, 209, 178, 0.1)',
                                tension: 0.3,
                                fill: true
                            },
                            {
                                label: 'Saldo Inversión',
                                data: saldosInversion,
                                borderColor: 'rgba(0, 184, 255, 1)',
                                backgroundColor: 'rgba(0, 184, 255, 0.1)',
                                tension: 0.3,
                                fill: true
                            },
                            {
                                label: 'Ganancias',
                                data: saldosGanancias,
                                borderColor: 'rgba(255, 221, 87, 1)',
                                backgroundColor: 'rgba(255, 221, 87, 0.1)',
                                tension: 0.3,
                                fill: true
                            }
                        ]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'top',
                                labels: {
                                    color: '#f0f0f0'
                                }
                            },
                            tooltip: {
                                mode: 'index',
                                intersect: false,
                                callbacks: {
                                    label: function(context) {
                                        return context.dataset.label + ': $' + context.raw.toFixed(2) + ' USDT';
                                    }
                                }
                            }
                        },
                        scales: {
                            x: {
                                grid: {
                                    color: 'rgba(255, 255, 255, 0.1)'
                                },
                                ticks: {
                                    color: '#f0f0f0'
                                }
                            },
                            y: {
                                grid: {
                                    color: 'rgba(255, 255, 255, 0.1)'
                                },
                                ticks: {
                                    color: '#f0f0f0',
                                    callback: function(value) {
                                        return '$' + value.toFixed(2);
                                    }
                                }
                            }
                        },
                        interaction: {
                            mode: 'nearest',
                            axis: 'x',
                            intersect: false
                        }
                    }
                });
            }
        });
    </script>
</body>
</html>