<?php
include 'connect.php';

// Establecer zona horaria correcta
date_default_timezone_set('UTC');

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM usuarios WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

// Función para calcular días transcurridos desde la creación del plan
function diasDesdeCreacion($fecha_inicio) {
    $inicio = new DateTime($fecha_inicio);
    $hoy = new DateTime('now', new DateTimeZone('UTC'));
    return $inicio->diff($hoy)->days;
}

// Función para calcular ganancias acumuladas automáticamente
function calcularGananciasAutomaticas($conn, $user_id) {
    // Obtener todos los planes activos del usuario
    $stmt = $conn->prepare("SELECT * FROM planes WHERE usuario_id = ? AND completado = FALSE");
    $stmt->execute([$user_id]);
    $planes = $stmt->fetchAll();
    
    $ganancias_totales = 0;
    
    foreach ($planes as $plan) {
        $ultima_ganancia = strtotime($plan['ultima_ganancia']);
        $ahora = time();
        $diferencia = $ahora - $ultima_ganancia;
        
        // Calcular cuántos ciclos de 24 horas han pasado
        $ciclos_completos = floor($diferencia / (24 * 3600));
        
        if ($ciclos_completos > 0) {
            // Calcular ganancia por cada ciclo completo
            $ganancia = $plan['monto_actual'] * 0.01 * $ciclos_completos;
            
            // Actualizar el plan
            $nueva_fecha = date('Y-m-d H:i:s', $ultima_ganancia + ($ciclos_completos * 24 * 3600));
            $dias_transcurridos = $plan['dias_transcurridos'] + $ciclos_completos;
            $porcentaje_completado = min(($dias_transcurridos / 210) * 200, 200);
            $completado = ($porcentaje_completado >= 200) ? 1 : 0;
            
            $stmt = $conn->prepare("UPDATE planes SET 
                ganancias_pendientes = ganancias_pendientes + ?,
                dias_transcurridos = ?,
                porcentaje_completado = ?,
                completado = ?,
                ultima_ganancia = ?,
                fecha_ultima_actualizacion = UTC_TIMESTAMP()
                WHERE id = ?");
            $stmt->execute([$ganancia, $dias_transcurridos, $porcentaje_completado, $completado, $nueva_fecha, $plan['id']]);
            
            $ganancias_totales += $ganancia;
        }
    }
    
    return $ganancias_totales;
}

// Calcular ganancias automáticas al cargar la página
calcularGananciasAutomaticas($conn, $user_id);

// Función para calcular ganancias en tiempo real
function calcularGananciasTiempoReal($plan) {
    $ultima_ganancia = strtotime($plan['ultima_ganancia']);
    $ahora = time();
    $diferencia = $ahora - $ultima_ganancia;
    
    // Calcular porcentaje completado del ciclo actual (0 a 1)
    $porcentaje_ciclo = min($diferencia / (24 * 3600), 1);
    
    // Calcular ganancia acumulada en este ciclo
    $ganancia_ciclo_actual = $plan['monto_actual'] * 0.01 * $porcentaje_ciclo;
    
    // Ganancia total (pendientes + ciclo actual)
    $ganancia_total = $plan['ganancias_pendientes'] + $ganancia_ciclo_actual;
    
    return [
        'ganancia_actual' => $ganancia_ciclo_actual,
        'ganancia_total' => $ganancia_total,
        'porcentaje_ciclo' => $porcentaje_ciclo * 100,
        'proximo_ciclo' => $ultima_ganancia + (24 * 3600),
        'segundos_restantes' => max(0, ($ultima_ganancia + (24 * 3600)) - $ahora)
    ];
}

// Procesar acciones sobre los planes
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['invertir'])) {
        // Procesar nueva inversión
        $monto = floatval($_POST['monto']);
        
        if ($monto >= 10 && $monto <= $user['saldo_inversion']) {
            // Crear nuevo plan con tiempo exacto UTC
            $fecha_actual = gmdate('Y-m-d H:i:s');
            $stmt = $conn->prepare("INSERT INTO planes (
                usuario_id, 
                monto_inicial, 
                monto_actual, 
                fecha_proximo_pago,
                ultima_ganancia,
                fecha_inicio
            ) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$user_id, $monto, $monto, $fecha_actual, $fecha_actual, $fecha_actual]);
            
            // Actualizar saldo de inversión
            $nuevo_saldo = $user['saldo_inversion'] - $monto;
            $stmt = $conn->prepare("UPDATE usuarios SET saldo_inversion = ? WHERE id = ?");
            $stmt->execute([$nuevo_saldo, $user_id]);
            
            header('Location: planes.php?success=Inversión creada correctamente');
            exit();
        } else {
            $error = $monto < 10 ? 'El monto mínimo es $10' : 'Saldo insuficiente';
        }
    } 
    elseif (isset($_POST['reinvertir'])) {
        // Procesar reinversión en un plan existente
        $plan_id = intval($_POST['plan_id']);
        $monto = floatval($_POST['monto_reinversion']);
        
        // Verificar que el plan existe y pertenece al usuario
        $stmt = $conn->prepare("SELECT * FROM planes WHERE id = ? AND usuario_id = ?");
        $stmt->execute([$plan_id, $user_id]);
        $plan = $stmt->fetch();
        
        if ($plan && !$plan['completado'] && $monto >= 0.25 && $monto <= $user['saldo_planes']) {
            // Actualizar el plan
            $nuevo_monto = $plan['monto_actual'] + $monto;
            $fecha_actual = gmdate('Y-m-d H:i:s');
            
            $stmt = $conn->prepare("UPDATE planes SET 
                monto_actual = ?, 
                fecha_ultima_actualizacion = UTC_TIMESTAMP(),
                es_reinversion = TRUE
                WHERE id = ?");
            $stmt->execute([$nuevo_monto, $plan_id]);
            
            // Actualizar saldo de planes
            $nuevo_saldo = $user['saldo_planes'] - $monto;
            $stmt = $conn->prepare("UPDATE usuarios SET saldo_planes = ? WHERE id = ?");
            $stmt->execute([$nuevo_saldo, $user_id]);
            
            header('Location: planes.php?success=Reinversión realizada correctamente');
            exit();
        } else {
            $error = 'No se pudo realizar la reinversión. El monto mínimo es $0.25 o verifica el estado del plan.';
        }
    }
    elseif (isset($_POST['recolectar'])) {
        // Procesar recolección de ganancias
        $plan_id = intval($_POST['plan_id']);
        
        // Verificar que el plan existe y pertenece al usuario
        $stmt = $conn->prepare("SELECT * FROM planes WHERE id = ? AND usuario_id = ?");
        $stmt->execute([$plan_id, $user_id]);
        $plan = $stmt->fetch();
        
        if ($plan && !$plan['completado']) {
            // Calcular ganancias pendientes + ganancias del ciclo actual
            $ganancias_realtime = calcularGananciasTiempoReal($plan);
            $ganancia_total = $ganancias_realtime['ganancia_total'];
            
            if ($ganancia_total > 0) {
                // Actualizar saldo de planes del usuario
                $nuevo_saldo = $user['saldo_planes'] + $ganancia_total;
                $stmt = $conn->prepare("UPDATE usuarios SET saldo_planes = ? WHERE id = ?");
                $stmt->execute([$nuevo_saldo, $user_id]);
                
                // Resetear ganancias pendientes en el plan y actualizar última ganancia
                $nueva_fecha = gmdate('Y-m-d H:i:s');
                $stmt = $conn->prepare("UPDATE planes SET 
                    ganancia_acumulada = ganancia_acumulada + ?,
                    ganancias_pendientes = 0,
                    ultima_ganancia = ?,
                    fecha_ultima_actualizacion = UTC_TIMESTAMP()
                    WHERE id = ?");
                $stmt->execute([$ganancia_total, $nueva_fecha, $plan_id]);
                
                header('Location: planes.php?success=Ganancia recolectada correctamente');
                exit();
            } else {
                $error = 'No hay ganancias disponibles para recolectar en este momento';
            }
        } else {
            $error = 'El plan no existe o está completado';
        }
    }
}

// Obtener planes activos con cálculo de tiempo para el próximo ciclo
$stmt = $conn->prepare("SELECT *, 
    TIMESTAMPDIFF(SECOND, ultima_ganancia, UTC_TIMESTAMP()) AS segundos_desde_ultima_ganancia
    FROM planes 
    WHERE usuario_id = ? AND completado = FALSE 
    ORDER BY fecha_inicio DESC");
$stmt->execute([$user_id]);
$planes_activos = $stmt->fetchAll();

// Obtener planes completados
$stmt = $conn->prepare("SELECT * FROM planes WHERE usuario_id = ? AND completado = TRUE ORDER BY fecha_inicio DESC LIMIT 5");
$stmt->execute([$user_id]);
$planes_completados = $stmt->fetchAll();

// Función para calcular el tiempo restante para el próximo ciclo
function tiempoRestante($ultima_ganancia) {
    $ahora = time();
    $proximo_ciclo = strtotime($ultima_ganancia) + (24 * 3600);
    
    if ($ahora >= $proximo_ciclo) {
        return "00:00:00";
    }
    
    $diferencia = $proximo_ciclo - $ahora;
    $horas = floor($diferencia / 3600);
    $minutos = floor(($diferencia % 3600) / 60);
    $segundos = $diferencia % 60;
    
    return sprintf("%02d:%02d:%02d", $horas, $minutos, $segundos);
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>JC Investment - Planes</title>
    <style>
        :root {
            --primary: #00ff9d;
            --secondary: #00b8ff;
            --dark: #121212;
            --darker: #0a0a0a;
            --light: #f0f0f0;
            --danger: #ff3860;
            --success: #00d1b2;
            --warning: #ffdd57;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        body {
            background-color: var(--dark);
            color: var(--light);
            display: flex;
            min-height: 100vh;
        }
        
        .sidebar {
            width: 250px;
            background-color: rgba(10, 10, 10, 0.8);
            border-right: 1px solid rgba(0, 255, 157, 0.1);
            padding: 1.5rem;
            position: fixed;
            height: 100vh;
            z-index: 100;
        }
        
        .main-content {
            flex: 1;
            margin-left: 250px;
            padding: 2rem;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid rgba(0, 255, 157, 0.1);
        }
        
        .user-info {
            display: flex;
            align-items: center;
        }
        
        .user-name {
            font-size: 1.2rem;
            font-weight: bold;
        }
        
        .logout-btn {
            background-color: rgba(255, 56, 96, 0.1);
            color: var(--danger);
            border: none;
            padding: 0.5rem 1rem;
            border-radius: 5px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .section-title {
            font-size: 1.2rem;
            margin-bottom: 1rem;
            color: var(--primary);
            display: flex;
            align-items: center;
        }
        
        .section-title i {
            margin-right: 0.5rem;
        }
        
        .investment-container {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 2rem;
            margin-bottom: 2rem;
        }
        
        .investment-card {
            background-color: rgba(10, 10, 10, 0.8);
            border: 1px solid rgba(0, 255, 157, 0.1);
            border-radius: 10px;
            padding: 1.5rem;
        }
        
        .investment-card h3 {
            margin-bottom: 1rem;
            color: var(--primary);
        }
        
        .investment-card p {
            margin-bottom: 1rem;
            color: rgba(255, 255, 255, 0.7);
        }
        
        .form-group {
            margin-bottom: 1rem;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            color: var(--light);
        }
        
        .form-group input {
            width: 100%;
            padding: 0.8rem;
            background-color: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(0, 255, 157, 0.3);
            border-radius: 5px;
            color: var(--light);
        }
        
        .btn {
            padding: 0.8rem 1.5rem;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            border: none;
            border-radius: 5px;
            color: var(--dark);
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 255, 157, 0.3);
        }
        
        .btn-small {
            padding: 0.5rem 1rem;
            font-size: 0.8rem;
        }
        
        .btn-success {
            background: var(--success);
        }
        
        .btn-warning {
            background: var(--warning);
            color: var(--dark);
        }
        
        .error {
            color: var(--danger);
            margin-bottom: 1rem;
            padding: 0.5rem;
            background-color: rgba(255, 56, 96, 0.1);
            border-radius: 5px;
        }
        
        .success {
            color: var(--success);
            margin-bottom: 1rem;
            padding: 0.5rem;
            background-color: rgba(0, 209, 178, 0.1);
            border-radius: 5px;
        }
        
        .plan-card {
            background-color: rgba(10, 10, 10, 0.8);
            border: 1px solid rgba(0, 255, 157, 0.1);
            border-radius: 10px;
            padding: 1rem;
            margin-bottom: 1rem;
            position: relative;
            overflow: hidden;
        }
        
        .plan-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 3px;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
        }
        
        .plan-card.reinvestment::before {
            background: linear-gradient(90deg, var(--secondary), var(--primary));
        }
        
        .plan-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
        }
        
        .plan-title {
            font-weight: bold;
            color: var(--primary);
        }
        
        .plan-reinvestment .plan-title {
            color: var(--secondary);
        }
        
        .plan-date {
            font-size: 0.8rem;
            color: rgba(255, 255, 255, 0.5);
        }
        
        .plan-amount {
            margin-bottom: 0.5rem;
        }
        
        .progress-container {
            width: 100%;
            height: 10px;
            background-color: rgba(255, 255, 255, 0.1);
            border-radius: 5px;
            margin-bottom: 0.5rem;
            overflow: hidden;
        }
        
        .progress-bar {
            height: 100%;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            border-radius: 5px;
            transition: width 0.5s ease;
        }
        
        .plan-stats {
            display: flex;
            justify-content: space-between;
            font-size: 0.8rem;
            color: rgba(255, 255, 255, 0.7);
        }
        
        .plan-actions {
            display: flex;
            gap: 0.5rem;
            margin-top: 1rem;
        }
        
        .timer-container {
            display: flex;
            flex-direction: column;
            align-items: center;
            margin: 1rem 0;
        }
        
        .timer-circle {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            border: 3px solid var(--primary);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto;
            font-weight: bold;
            color: var(--primary);
            font-family: monospace;
            font-size: 1.2rem;
            position: relative;
            margin-bottom: 0.5rem;
        }
        
        .timer-circle::after {
            content: '';
            position: absolute;
            width: 100%;
            height: 100%;
            border-radius: 50%;
            border: 3px solid transparent;
            border-top-color: var(--secondary);
            animation: spin 1s linear infinite;
        }
        
        .timer-label {
            font-size: 0.8rem;
            color: rgba(255, 255, 255, 0.7);
        }
        
        .real-time-earning {
            margin-top: 0.5rem;
            text-align: center;
        }
        
        .earning-progress {
            width: 100%;
            height: 6px;
            background-color: rgba(255, 255, 255, 0.1);
            border-radius: 3px;
            margin: 0.5rem 0;
            overflow: hidden;
        }
        
        .earning-progress-bar {
            height: 100%;
            background-color: var(--success);
            width: 0%;
            transition: width 1s linear;
        }
        
        .earning-amount {
            font-size: 0.9rem;
            color: var(--success);
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .plans-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
        }
        
        .plans-section {
            margin-bottom: 2rem;
        }
        
        .plan-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 0.5rem;
            margin-top: 0.5rem;
        }
        
        .detail-item {
            background-color: rgba(0, 255, 157, 0.1);
            padding: 0.5rem;
            border-radius: 5px;
        }
        
        .detail-label {
            font-size: 0.7rem;
            color: rgba(255, 255, 255, 0.7);
        }
        
        .detail-value {
            font-weight: bold;
            font-size: 0.9rem;
        }

        .logo {
            text-align: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid rgba(0, 255, 157, 0.1);
        }
        
        .logo h2 {
            font-size: 1.5rem;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            -webkit-background-clip: text;
            background-clip: text;
            color: transparent;
            text-transform: uppercase;
            letter-spacing: 2px;
        }
        
        .nav-menu {
            list-style: none;
        }
        
        .nav-item {
            margin-bottom: 0.5rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 0.8rem 1rem;
            color: var(--light);
            text-decoration: none;
            border-radius: 5px;
            transition: all 0.3s ease;
        }
        
        .nav-link:hover, .nav-link.active {
            background-color: rgba(0, 255, 157, 0.1);
            color: var(--primary);
        }
        
        .nav-link i {
            margin-right: 0.8rem;
            font-size: 1.2rem;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: rgba(0, 255, 157, 0.1);
            margin-right: 1rem;
            overflow: hidden;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .logout-btn:hover {
            background-color: rgba(255, 56, 96, 0.2);
        }
        
        .saldo-info {
            background: rgba(0, 184, 255, 0.1);
            padding: 0.8rem;
            border-radius: 8px;
            margin-top: 1rem;
            border-left: 3px solid var(--secondary);
        }
        
        .saldo-info p {
            font-size: 0.9rem;
            margin-bottom: 0.5rem;
        }
        
        .saldo-info strong {
            color: var(--secondary);
        }

        .plan-days-info {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .plan-days-item {
            background-color: rgba(0, 255, 157, 0.1);
            padding: 0.3rem 0.5rem;
            border-radius: 5px;
            text-align: center;
            flex: 1;
            margin: 0 0.2rem;
        }
        
        .plan-days-label {
            font-size: 0.7rem;
            color: rgba(255, 255, 255, 0.7);
        }
        
        .plan-days-value {
            font-weight: bold;
            color: var(--primary);
        }

        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
            }
            
            .main-content {
                margin-left: 0;
            }
            
            .investment-container {
                grid-template-columns: 1fr;
            }
            
            .plans-grid {
                grid-template-columns: 1fr;
            }
            
            .plan-actions {
                flex-direction: column;
            }
            
            .plan-days-info {
                flex-direction: column;
                gap: 0.3rem;
            }
            
            .plan-days-item {
                margin: 0;
            }
        }
    </style>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
</head>
<body>
    <div class="sidebar">
        <div class="logo">
            <h2>JC Investment</h2>
        </div>
        
        <ul class="nav-menu">
            <li class="nav-item">
                <a href="dashboard.php" class="nav-link">
                    <i class="fas fa-home"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="perfil.php" class="nav-link">
                    <i class="fas fa-user"></i>
                    <span>Perfil</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="planes.php" class="nav-link active">
                    <i class="fas fa-chart-line"></i>
                    <span>Planes</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="transferencia.php" class="nav-link">
                    <i class="fas fa-exchange-alt"></i>
                    <span>Transferencia</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="historial.php" class="nav-link">
                    <i class="fas fa-history"></i>
                    <span>Historial</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="deposito.php" class="nav-link">
                    <i class="fas fa-wallet"></i>
                    <span>Depósito</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="retiro.php" class="nav-link">
                    <i class="fas fa-money-bill-wave"></i>
                    <span>Retiro</span>
                </a>
            </li>
        </ul>
    </div>
    
    <div class="main-content">
        <div class="header">
            <div class="user-info">
                <div class="user-avatar">
                    <img src="<?php echo isset($user['foto_perfil']) ? $user['foto_perfil'] : 'default.png'; ?>" alt="Foto de perfil">
                </div>
                <div class="user-name"><?php echo htmlspecialchars($user['nombre_completo']); ?></div>
            </div>
            <button class="logout-btn" onclick="window.location.href='logout.php'">
                <i class="fas fa-sign-out-alt"></i> Cerrar Sesión
            </button>
        </div>
        
        <?php if (isset($error)): ?>
            <div class="error">
                <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($_GET['success'])): ?>
            <div class="success">
                <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($_GET['success']); ?>
            </div>
        <?php endif; ?>
        
        <div class="investment-container">
            <div class="investment-card">
                <h3><i class="fas fa-plus-circle"></i> Nueva Inversión</h3>
                <p>Ingresa la cantidad que deseas invertir. Cada inversión generará un 1% diario hasta completar 200% en 210 días.</p>
                
                <form action="planes.php" method="POST">
                    <div class="form-group">
                        <label for="monto">Monto a invertir (USDT)</label>
                        <input type="number" id="monto" name="monto" min="10" max="<?php echo $user['saldo_inversion']; ?>" step="0.01" required>
                        <small>Saldo disponible: $<?php echo number_format($user['saldo_inversion'], 2); ?> USDT</small>
                    </div>
                    
                    <button type="submit" name="invertir" class="btn">Invertir</button>
                </form>
            </div>
            
            <div class="investment-card">
                <h3><i class="fas fa-info-circle"></i> Información de Planes</h3>
                <p><strong>Inversiones:</strong> Generan 1% diario durante 210 días (200% total).</p>
                <p><strong>Reinversiones:</strong> Sumas ganancias al capital con los mismos beneficios. Puedes reinvertir desde $0.25 USDT.</p>
                <p><strong>Recolección:</strong> Puedes recolectar tus ganancias acumuladas en cualquier momento.</p>
                
                <div class="plan-details">
                    <div class="detail-item">
                        <div class="detail-label">Disponible Para Inversión</div>
                        <div class="detail-value">$<?php echo number_format($user['saldo_inversion'], 2); ?></div>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">Reinvercion o Retiro</div>
                        <div class="detail-value">$<?php echo number_format($user['saldo_planes'], 2); ?></div>
                    </div>
                </div>
                
                <div class="saldo-info">
                    <p><i class="fas fa-info-circle"></i> <strong>Reinversión disponible:</strong></p>
                    <p>Puedes reinvertir cualquier monto a partir de $0.25 USDT usando tus ganancias acumuladas.</p>
                </div>
            </div>
        </div>
        
        <div class="plans-grid">
            <div class="plans-section">
                <h3 class="section-title">
                    <i class="fas fa-spinner"></i> Planes Activos
                </h3>
                
                <?php if (empty($planes_activos)): ?>
                    <p>No tienes planes activos actualmente.</p>
                <?php else: ?>
                    <?php foreach ($planes_activos as $plan): 
                        $ganancias_realtime = calcularGananciasTiempoReal($plan);
                        $dias_transcurridos = diasDesdeCreacion($plan['fecha_inicio']);
                        $dias_restantes = max(0, 210 - $dias_transcurridos);
                    ?>
                        <div class="plan-card <?php echo $plan['es_reinversion'] ? 'reinvestment' : ''; ?>">
                            <div class="plan-header">
                                <div class="plan-title">
                                    <?php echo $plan['es_reinversion'] ? 'Plan de Reinversión' : 'Plan de Inversión'; ?>
                                </div>
                                <div class="plan-date">
                                    <?php echo date('d/m/Y H:i', strtotime($plan['fecha_inicio'])); ?> UTC
                                </div>
                            </div>
                            
                            <div class="plan-amount">
                                $<?php echo number_format($plan['monto_inicial'], 2); ?> USDT
                                <small>(Actual: $<?php echo number_format($plan['monto_actual'], 2); ?>)</small>
                            </div>
                            
                            <div class="plan-days-info">
                                <div class="plan-days-item">
                                    <div class="plan-days-label">Días transcurridos</div>
                                    <div class="plan-days-value"><?php echo $dias_transcurridos; ?></div>
                                </div>
                                <div class="plan-days-item">
                                    <div class="plan-days-label">Días restantes</div>
                                    <div class="plan-days-value"><?php echo $dias_restantes; ?></div>
                                </div>
                            </div>
                            
                            <div class="progress-container">
                                <div class="progress-bar" style="width: <?php echo min(($dias_transcurridos / 210) * 100, 100); ?>%"></div>
                            </div>
                            
                            <div class="plan-stats">
                                <span>Ganancia diaria: $<?php echo number_format($plan['monto_actual'] * 0.01, 2); ?></span>
                                <span>Progreso: <?php echo number_format(($dias_transcurridos / 210) * 100, 2); ?>%</span>
                            </div>
                            
                            <div class="timer-container">
                                <div class="timer-circle" id="timer-<?php echo $plan['id']; ?>">
                                    <?php echo tiempoRestante($plan['ultima_ganancia']); ?>
                                </div>
                                <div class="timer-label">Próximo ciclo en</div>
                                
                                <div class="real-time-earning">
                                    <div class="earning-progress">
                                        <div class="earning-progress-bar" id="progress-<?php echo $plan['id']; ?>" style="width: <?php echo $ganancias_realtime['porcentaje_ciclo']; ?>%"></div>
                                    </div>
                                    <div class="earning-amount">
                                        Ganancia actual: $<span id="earning-<?php echo $plan['id']; ?>"><?php echo number_format($ganancias_realtime['ganancia_actual'], 4); ?></span>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="plan-actions">
                                <form action="planes.php" method="POST" style="flex: 1;">
                                    <input type="hidden" name="plan_id" value="<?php echo $plan['id']; ?>">
                                    <button type="submit" name="recolectar" class="btn btn-success btn-small">
                                        <i class="fas fa-coins"></i> Recolectar $<span id="pending-<?php echo $plan['id']; ?>"><?php echo number_format($ganancias_realtime['ganancia_total'], 4); ?></span>
                                    </button>
                                </form>
                                
                                <form action="planes.php" method="POST" style="flex: 1;">
                                    <input type="hidden" name="plan_id" value="<?php echo $plan['id']; ?>">
                                    <div class="form-group" style="margin-bottom: 0.5rem;">
                                        <input type="number" name="monto_reinversion" min="0.25" max="<?php echo $user['saldo_planes']; ?>" step="0.01" placeholder="Monto reinversión" required style="padding: 0.5rem; font-size: 0.9rem;">
                                    </div>
                                    <button type="submit" name="reinvertir" class="btn btn-warning btn-small">
                                        <i class="fas fa-sync-alt"></i> Reinvertir
                                    </button>
                                </form>
                            </div>
                            
                            <div class="plan-details">
                                <div class="detail-item">
                                    <div class="detail-label">Ganancia Acumulada</div>
                                    <div class="detail-value">$<?php echo number_format($plan['ganancia_acumulada'], 2); ?></div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Próxima Ganancia</div>
                                    <div class="detail-value">$<?php echo number_format($plan['monto_actual'] * 0.01, 2); ?></div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            
            <div class="plans-section">
                <h3 class="section-title">
                    <i class="fas fa-check-circle"></i> Planes Completados
                </h3>
                
                <?php if (empty($planes_completados)): ?>
                    <p>No has completado ningún plan todavía.</p>
                <?php else: ?>
                    <?php foreach ($planes_completados as $plan): ?>
                        <div class="plan-card <?php echo $plan['es_reinversion'] ? 'reinvestment' : ''; ?>">
                            <div class="plan-header">
                                <div class="plan-title">
                                    <?php echo $plan['es_reinversion'] ? 'Reinversión Completada' : 'Inversión Completada'; ?>
                                </div>
                                <div class="plan-date">
                                    <?php echo date('d/m/Y H:i', strtotime($plan['fecha_inicio'])); ?> UTC
                                </div>
                            </div>
                            
                            <div class="plan-amount">
                                $<?php echo number_format($plan['monto_inicial'], 2); ?> USDT
                            </div>
                            
                            <div class="progress-container">
                                <div class="progress-bar" style="width: 100%"></div>
                            </div>
                            
                            <div class="plan-stats">
                                <span>Días completados: 210/210</span>
                                <span>200% completado</span>
                            </div>
                            
                            <div class="plan-details">
                                <div class="detail-item">
                                    <div class="detail-label">Ganancia Total</div>
                                    <div class="detail-value">$<?php echo number_format($plan['ganancia_acumulada'], 2); ?></div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Retorno Total</div>
                                    <div class="detail-value">$<?php echo number_format($plan['monto_inicial'] * 2, 2); ?></div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        // Datos iniciales para cada plan
        const plansData = {};
        <?php foreach ($planes_activos as $plan): 
            $ganancias = calcularGananciasTiempoReal($plan);
        ?>
        plansData[<?php echo $plan['id']; ?>] = {
            montoActual: <?php echo $plan['monto_actual']; ?>,
            gananciasPendientes: <?php echo $plan['ganancias_pendientes']; ?>,
            ultimaGanancia: <?php echo strtotime($plan['ultima_ganancia']) * 1000; ?>,
            nextCycle: <?php echo ($ganancias['proximo_ciclo'] * 1000); ?>,
            earningRate: <?php echo $plan['monto_actual'] * 0.01 / (24 * 3600); ?>, // USDT por segundo
            gananciaActual: <?php echo $ganancias['ganancia_actual']; ?>,
            gananciaTotal: <?php echo $ganancias['ganancia_total']; ?>,
            porcentajeCiclo: <?php echo $ganancias['porcentaje_ciclo']; ?>
        };
        <?php endforeach; ?>

        // Función para actualizar ganancias en tiempo real
        function updateRealTimeEarnings() {
            const now = Date.now();
            
            Object.keys(plansData).forEach(planId => {
                const plan = plansData[planId];
                const elapsed = (now - plan.ultimaGanancia) / 1000; // segundos
                
                // Si ya pasó el ciclo, no actualizar (se recargará la página)
                if (now >= plan.nextCycle) return;
                
                // Calcular ganancia del ciclo actual
                const currentEarning = Math.min(elapsed * plan.earningRate, plan.montoActual * 0.01);
                const totalPending = plan.gananciasPendientes + currentEarning;
                const percent = (elapsed / (24 * 3600)) * 100;
                
                // Actualizar UI
                document.getElementById(`earning-${planId}`).textContent = currentEarning.toFixed(4);
                document.getElementById(`pending-${planId}`).textContent = totalPending.toFixed(4);
                document.getElementById(`progress-${planId}`).style.width = `${percent}%`;
            });
        }

        // Actualizar temporizadores en tiempo real
        function updateTimers() {
            const now = Date.now();
            
            Object.keys(plansData).forEach(planId => {
                const plan = plansData[planId];
                const nextCycleMs = plan.nextCycle;
                const timer = document.getElementById(`timer-${planId}`);
                
                // Calcular diferencia en milisegundos
                const diffMs = nextCycleMs - now;
                
                if (diffMs <= 0) {
                    timer.textContent = "00:00:00";
                    // Recargar la página para actualizar las ganancias
                    if (!timer.hasAttribute('data-reloading')) {
                        timer.setAttribute('data-reloading', 'true');
                        setTimeout(() => {
                            window.location.reload();
                        }, 5000); // Recargar después de 5 segundos
                    }
                } else {
                    // Calcular horas, minutos y segundos exactos
                    const totalSeconds = Math.floor(diffMs / 1000);
                    const hours = Math.floor(totalSeconds / 3600);
                    const minutes = Math.floor((totalSeconds % 3600) / 60);
                    const seconds = Math.floor(totalSeconds % 60);
                    
                    timer.textContent = `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                }
            });
        }
        
        // Actualizar cada segundo
        updateTimers();
        updateRealTimeEarnings();
        setInterval(() => {
            updateTimers();
            updateRealTimeEarnings();
        }, 1000);
        
        // Confirmación antes de reinvertir
        document.querySelectorAll('button[name="reinvertir"]').forEach(btn => {
            btn.addEventListener('click', function(e) {
                const amountInput = this.closest('form').querySelector('input[name="monto_reinversion"]');
                const amount = parseFloat(amountInput.value);
                
                if (amount < 0.25) {
                    alert('El monto mínimo de reinversión es $0.25 USDT');
                    e.preventDefault();
                    return;
                }
                
                if (!confirm(`¿Estás seguro de que deseas reinvertir $${amount.toFixed(2)} USDT en este plan?`)) {
                    e.preventDefault();
                }
            });
        });
    </script>
</body>
</html>